;;; -*- Mode:Emacs-Lisp -*-

;;; file: exit-minibuffer.el
;;; created: 10-Apr-91 by Jamie Zawinski <jwz@lucid.com>
;;;
;;; The code in this file modifies things such that when you are being prompted
;;; for a string with completion, and it is permitted to type a string which is
;;; not one of the known completions, an additional RET will be necessary to
;;; confirm that.  I find that with this behaviour, I almost never visit
;;; nonexistent files or switch to nonexistent buffers when I don't want to.
;;;
;;; Suppose you execute switch-to-buffer, and quickly type ``f o o TAB RET''
;;; expecting the TAB to fully complete to the buffer name, "foobar.c".  But
;;; you have forgotten that "foobar.h" is also in a buffer.  With the old
;;; system, you would now be sitting in the buffer "foobar.", but with this
;;; code, the minibuffer will be displaying
;;;
;;; 	Switch to buffer: foobar. [incomplete, confirm]
;;;
;;; if you really want this, you can just hit RET.  If you don't, keep typing.
;;;
;;; Similar behaviour will be exhibited for find-file, and just about anything
;;; else that does completion.  You can turn this code off completely by
;;; setting the variable `minibuffer-confirm-incomplete' to nil.

(defvar minibuffer-confirm-incomplete t
  "*If true, then in contexts where completing-read allows answers which
are not valid completions, an extra RET must be typed to confirm the
response.  This is helpful for catching typos, etc.")

(defun exit-minibuffer ()
  "Terminate this minibuffer argument.
If minibuffer-confirm-incomplete is true, and we are in a completing-read
of some kind, and the contents of the minibuffer is not an existing completion,
requires an additional RET before the minibuffer will be exited."
  (interactive)
  (let* ((input (minibuffer-contents))
	 (comp (if (or (null minibuffer-confirm-incomplete)
		       (null minibuffer-completion-table)
                       (string-equal input "")
                       (= (point-min) (point-max)))
		   t
		 (try-completion input
				 minibuffer-completion-table
				 minibuffer-completion-predicate))))
    (cond ((or (eq comp t)
	       (and comp
                    (cond ((consp minibuffer-completion-table)
                           (assoc input minibuffer-completion-table))
                          ((vectorp minibuffer-completion-table)
                           (intern-soft input minibuffer-completion-table))
                          (t (funcall minibuffer-completion-table
                                      input
                                      minibuffer-completion-predicate
                                      'lambda))) ))
	   (throw 'exit nil))
	  (t
	   (save-excursion
	     (let ((p (point-max)))
	       (goto-char p)
	       (unwind-protect
		    (progn
		      (save-excursion
			(or (= (preceding-char) ? ) (insert ? ))
			(insert (if (null comp)
				    "[no completions, confirm]"
                                  "[incomplete, confirm]")))
		      (let (c)
			(let ((inhibit-quit t))
			  (setq c (read-char)
				quit-flag nil))
			(cond ((= c ?\r)  (throw 'exit nil))
			      ((= c ?\^g) (throw 'exit t))
                              (t (setq unread-command-events c)))))
		 (delete-region p (point-max)))))) )))

(provide 'exit-minibuffer)
