// file      : CXX/Source.cxx
// author    : Boris Kolpackov <boris@dre.vanderbilt.edu>
// cvs-id    : $Id$

#include <CXX/Source.hpp>
#include <CXX/Elements.hpp>

#include <XSC/SemanticGraph.hpp>
#include <XSC/Traversal.hpp>

namespace
{
  struct Element : Traversal::Element, protected virtual Context
  {
    Element (Context& c)
        : Context (c)
    {
    }

    virtual void
    traverse (Type& c)
    {
      string name (c.name ());
      string type (type_name (c));

      //@@ can I get rid of the temporary?
      //

      //@@ need to use FQ-names.
      //
      os << "if (n == " << L << "\"" << name << "\")"
         << "{";

      if (c.max () != 1)
      {
        // sequence
        //
        os << type << " t (e);"
           << "add_" << name << " (t);";
      }
      else if (c.min () == 0)
      {
        // optional
        //
        os << type << " t (e);"
           << id (name) << " (t);";
      }
      else
      {
        // one
        //
        os << id (name) << "_ = ::std::auto_ptr< " << type << " > (" <<
          "new " << type << " (e));"
           << id (name) << "_->container (this);";
      }

      os <<"}"
         << "else ";
    }
  };


  struct Attribute : Traversal::Attribute, protected virtual Context
  {
    Attribute (Context& c)
        : Context (c)
    {
    }

    virtual void
    traverse (Type& a)
    {
      string name (a.name ());
      string type (type_name (a));

      //@@ need to use FQ-names.
      //
      os << "if (n == " << L << "\"" << name << "\")"
         << "{";

      if (a.optional ())
      {
        os << type << " t (a);"
           << id (name) << " (t);";
      }
      else
      {
        os << id (name) << "_ = ::std::auto_ptr< " << type << " > (" <<
          "new " << type << " (a));"
           << id (name) << "_->container (this);";
      }

      os << "}"
         << "else ";
    }
  };


  //@@ I am using this traverser to handle anonymous types which is
  //   not very clean (and asymmetric).
  //
  struct ComplexElement : Traversal::Complex, protected virtual Context
  {
    ComplexElement (Context& c,
                    Traversal::NodeDispatcher& anonymous_type,
                    string const& name_ = L"")
        : Context (c),
          name (name_),
          element_ (c),
          attribute_ (c)
    {
      names_elements_.node_traverser (element_);
      names_attributes_.node_traverser (attribute_);

      names_anonymous_.node_traverser (anonymous_type);
    }


    virtual void
    traverse (Type& c)
    {
      if (c.named ()) name = id (c.name ());

      enter_scope (name);

      c.context ().set ("name", name);

      // Go after anonymous types first.
      //
      names (c, names_anonymous_);

      Traversal::Complex::traverse (c);

      c.context ().remove ("name");

      leave_scope ();
    }

    virtual void
    pre (Type& c)
    {
      string type (type_name (c));

      os << "// " << scope << endl
         << "//" << endl
         << endl;

      os << scope << "::" << endl
         << name << " (" << xml_element_type << " const& e)" << endl
         << ":" << "Base__ (e), regulator__ ()"
         << "{"
         << endl;
    }

    using Traversal::Complex::names;

    virtual void
    names (Type& c)
    {
      bool he (has<Traversal::Element> (c));
      bool ha (has<Traversal::Attribute> (c));

      if (he || ha)
      {
        os << parser_type << " p (e);"
           << endl;

        if (he)
        {
          os << "while (p.more_elements ())"
             << "{"
             << xml_element_type << " e (p.next_element ());"
             << string_type << " n (::XSCRT::XML::uq_name (e.name ()));"
             << endl;

          names (c, names_elements_);

          os << "{}"        // else block
             << "}";        // while loop
        }

        if (ha)
        {
          os << "while (p.more_attributes ())"
             << "{"
             << xml_attribute_type << " a (p.next_attribute ());"
             << string_type << " n (::XSCRT::XML::uq_name (a.name ()));";

          names (c, names_attributes_);

          os << "{}"        // else block
             << "}";        // while loop
        }
      }
    }

    virtual void
    post (Type& c)
    {
      os << "}";
    }

  private:
    string name;

    Traversal::Names names_elements_;
    Traversal::Names names_attributes_;

    Element element_;
    Attribute attribute_;

    Traversal::Names names_anonymous_;
  };



  struct ComplexAttribute : Traversal::Complex, protected virtual Context
  {
    ComplexAttribute (Context& c, string const name_ = L"")
        : Context (c), name (name_)
    {
    }

    virtual void
    traverse (Type& c)
    {
      bool r (true);

      struct Traverser : Traversal::Inherits,
                         Traversal::Complex
      {
        Traverser (bool& v)
            : v_ (v)
        {
          edge_traverser (*this);
          node_traverser (*this);
        }

        virtual void
        traverse (SemanticGraph::Complex& c)
        {
          if (has<Traversal::Element> (c) ||
              has<Traversal::Attribute> (c)) v_ = false;
          else
          {
            inherits (c, *this);
          }
        }

      private:
        bool& v_;
      } t (r);

      t.traverse (c);

      if (r)
      {
        if (c.named ()) name = id (c.name ());

        enter_scope (name);

        c.context ().set ("name", scope);

        Traversal::Complex::traverse (c);

        c.context ().remove ("name");

        leave_scope ();
      }
    }


    virtual void
    pre (Type& c)
    {
      os << scope << "::" << endl
         << name << " (" << xml_attribute_type << " const& a)" << endl
         << ":" << endl
         << "Base__ (a)," << endl
         << "regulator__ ()"
         << "{";
    }

    virtual void
    post (Type& c)
    {
      os << "}";
    }

  private:
    string name;
  };

  struct StaticEnumerator : Traversal::Enumerator, protected virtual Context
  {
    StaticEnumerator (Context& c)
        : Context (c)
    {
    }

    virtual void
    traverse (Type& e)
    {
      string name (id (e.name ()));

      bool anon (!e.type ().named ());

      os << (anon ? "class " : "" ) << scope << " const " <<  scope << "::"
         << name << " (" << scope << "::" << name << "_l);";
    }
  };

  struct Enumerator : Traversal::Enumerator, protected virtual Context
  {
    Enumerator (Context& c)
        : Context (c)
    {
    }

    virtual void
    traverse (Type& e)
    {
      string name (e.name ());

      os << "if (v == " << L << "\"" << name << "\") v_ = " << id (name) << "_l;"
         << "else ";
    }
  };


  struct Enumeration : Traversal::Enumeration, protected virtual Context
  {
    Enumeration (Context& c, string name_ = L"")
        : Context (c), name (name_), enumerator_ (c), static_enumerator_ (c)
    {
      names_.node_traverser (enumerator_);
      names_static_.node_traverser (static_enumerator_);
    }

    virtual void
    traverse (Type& e)
    {
      if (e.named ()) name = id (e.name ());

      enter_scope (name);

      os << "// " << scope << endl
         << "//" << endl
         << endl;

      // c-tor (Element const&)
      //
      os << scope << "::" << endl
         << name << " (" << xml_element_type << " const& e)" << endl
         << ": Type (e)" << endl
         << "{"
         << string_type << " v (e.value ());"
         << endl;

      names (e, names_);

      os << "{}" // else block
         << "}";

      // c-tor (Attribute const&)
      //
      os << scope << "::" << endl
         << name << " (" << xml_attribute_type << " const& a)" << endl
         << ": Type (a)" << endl
         << "{"
         << string_type << " v (a.value ());"
         << endl;

      names (e, names_);

      os << "{}" // else block
         << "}";

      names (e, names_static_);

      os << endl;

      leave_scope ();
    }

  private:
    string name;

    Traversal::Names names_;
    Enumerator enumerator_;

    Traversal::Names names_static_;
    StaticEnumerator static_enumerator_;
  };


  struct AnonymousType : Traversal::Element, protected virtual Context
  {
    AnonymousType (Context& c)
        : Context (c)
    {
    }

    virtual void
    traverse (Type& e)
    {
      SemanticGraph::Type& t (e.type ());

      if (!t.named () && !t.context ().count ("seen"))
      {
        string name (id (e.name ()));

        Traversal::Belongs belongs;
        ComplexElement complex_element (*this, *this, name);
        ComplexAttribute complex_attribute (*this, name);
        Enumeration enumeration (*this, name);

        belongs.node_traverser (complex_element);
        belongs.node_traverser (complex_attribute);
        belongs.node_traverser (enumeration);

        t.context ().set ("seen", true);

        Element::belongs (e, belongs);

        t.context ().remove ("seen");
      }
    }
  };
}

void
generate_source (Context& ctx, SemanticGraph::Schema& schema)
{
  Traversal::Schema traverser;
  Traversal::Sources sources;
  Traversal::Names schema_names;
  Namespace ns (ctx);

  traverser.edge_traverser (sources);
  traverser.edge_traverser (schema_names);
  sources.node_traverser (traverser);
  schema_names.node_traverser (ns);

  Traversal::Names names;
  AnonymousType anonymous_type (ctx);
  ComplexElement complex_element (ctx, anonymous_type);
  ComplexAttribute complex_attribute (ctx);
  Enumeration enumeration (ctx);

  ns.edge_traverser (names);
  names.node_traverser (enumeration);
  names.node_traverser (complex_element);
  names.node_traverser (complex_attribute);
  names.node_traverser (anonymous_type);

  traverser.traverse (schema);
}
