// file      : CXX/Header.cpp
// author    : Boris Kolpackov <boris@dre.vanderbilt.edu>
// cvs-id    : $Id$

#include <CXX/Header.hpp>
#include <CXX/Elements.hpp>

#include <XSC/SemanticGraph.hpp>
#include <XSC/Traversal.hpp>

#include <CCF/CodeGenerationKit/Regex.hpp>

namespace
{
  struct Base : Traversal::Type, protected virtual Context
  {
    Base (Context& c)
        : Context (c)
    {
    }

    virtual void
    traverse (SemanticGraph::Type& c)
    {
      string name (type_name (c));

      os << " : public " << name
         << "{"
         << "//@@ VC6 anathema" << endl
         << "typedef " << name << " Base__;"
         << endl;

    }
  };

  // List of all non-optional members
  //
  //
  struct CtorArgs : Traversal::Complex,
                    protected Traversal::Enumeration,
                    protected Traversal::Type,
                    protected Traversal::Element,
                    protected Traversal::Attribute,
                    protected virtual Context
  {
    CtorArgs (Context& c)
        : Context (c)
    {
      complex_.edge_traverser (inherits_);
      complex_.edge_traverser (names_);

      inherits_.node_traverser (*this);
      names_.node_traverser (*this);
    }

    virtual void
    traverse (SemanticGraph::Type& t)
    {
      os << comma () << type_name (t) << " const& b__";
    }

    virtual void
    traverse (SemanticGraph::Enumeration& e)
    {
      os << comma () << type_name (e) << " const& b__";
    }

    virtual void
    traverse (SemanticGraph::Element& e)
    {
      if (e.min () == 1 && e.max () == 1)
      {
        os << comma () << type_name (e) << " const& " << id (e.name ()) << "__";
      }
    }

    virtual void
    traverse (SemanticGraph::Attribute& a)
    {
      if (!a.optional ())
      {
        os << comma () << type_name (a) << " const& " << id (a.name ()) << "__";
      }
    }

    void
    traverse (SemanticGraph::Complex& c)
    {
      first_ = true;

      complex_.traverse (c);
    }

  private:
    string
    comma ()
    {
      return first_ ? (first_ = false, L"") : L",\n";
    }

    bool first_;

  private:
    Traversal::Complex complex_;

    Traversal::Inherits inherits_;
    Traversal::Names names_;
  };


  struct Element : Traversal::Element, protected virtual Context
  {
    Element (Context& c)
        : Context (c)
    {
    }

    string
    type_name (Type& i)
    {
      SemanticGraph::Type& t (i.type ());

      if (t.named ())
      {
        return Context::type_name (t);
      }
      else
      {
        return anon_prefix_ + id (i.name ()) + anon_suffix_;
      }
    }


    virtual void
    traverse (Type& c)
    {
      string name (c.name ());
      string type (type_name (c));

      os << "// " << name << endl
         << "// " << endl
         << "public:" << endl;

      if (c.max () != 1)
      {
        // sequence
        //
        os << "typedef ::std::vector< " << type << " >::iterator "
           << name << "_iterator;";

        os << "typedef ::std::vector< " << type << " >::const_iterator "
           << name << "_const_iterator;";

        os << name << "_iterator begin_" << name << " ();";
        os << name << "_iterator end_" << name << " ();";
        os << name << "_const_iterator begin_" << name << " () const;";
        os << name << "_const_iterator end_" << name << " () const;";

        os << "void add_" << name << " (" << type << " const& );";

        os << endl
           << "protected:" << endl;

        os << "::std::vector< " << type << " > " << name << "_;";
      }
      else if (c.min () == 0)
      {
        // optional
        //
        os << "bool " << name << "_p () const;";
        os << type << " const& " << id (name) << " () const;";
        os << type << "& " << id (name) << " ();";
        os << "void " << id (name) << " (" << type << " const& );";

        os << endl
           << "protected:" << endl;

        os << "::std::auto_ptr< " << type << " > " << id (name) << "_;";
      }
      else
      {
        // one
        //
        os << type << " const& " << id (name) << " () const;";
        os << type << "& " << id (name) << " ();";
        os << "void " << id (name) << " (" << type << " const& );";

        os << endl
           << "protected:" << endl;

        os << "::std::auto_ptr< " << type << " > " << id (name) << "_;";
      }

      os << endl;
    }
  };


  struct Attribute : Traversal::Attribute, protected virtual Context
  {
    Attribute (Context& c)
        : Context (c)
    {
    }

    string
    type_name (Type& i)
    {
      SemanticGraph::Type& t (i.type ());

      if (t.named ())
      {
        return Context::type_name (t);
      }
      else
      {
        return anon_prefix_ + id (i.name ()) + anon_suffix_;
      }
    }

    virtual void
    traverse (Type& a)
    {
      string name (a.name ());
      string type (type_name (a));

      os << "// " << name << endl
         << "// " << endl
         << "public:" << endl;


      if (a.optional ())
      {
        os << "bool " << name << "_p () const;";
        os << type << " const& " << id (name) << " () const;";
        os << type << "& " << id (name) << " ();";
        os << "void " << id (name) << " (" << type << " const& );";

        os << endl
           << "protected:" << endl;

        os << "::std::auto_ptr< " << type << " > " << id (name) << "_;";
      }
      else
      {
        os << type << " const& " << id (name) << " () const;";
        os << type << "& " << id (name) << " ();";
        os << "void " << id (name) << " (" << type << " const& );";

        os << endl
           << "protected:" << endl;

        os << "::std::auto_ptr< " << type << " > " << id (name) << "_;";
      }

      os << endl;
    }
  };


  struct Complex : Traversal::Complex, protected virtual Context
  {
    Complex (Context& c,
             Traversal::NodeDispatcher& anonymous_type,
             string const& name_ = L"")
        : Context (c),
          name (name_),
          e (c.esymbol.empty () ? c.esymbol : c.esymbol + L" "),
          base_ (c),
          element_ (c),
          attribute_ (c),
          ctor_args_ (c)
    {
      edge_traverser (inherits_);
      edge_traverser (names_);

      inherits_.node_traverser (base_);
      names_.node_traverser (anonymous_type);
      names_.node_traverser (element_);
      names_.node_traverser (attribute_);
    }

    virtual void
    traverse (Type& c)
    {
      if (c.named ()) name = id (c.name ());

      enter_scope (name);

      Traversal::Complex::traverse (c);

      leave_scope ();
    }

    virtual void
    pre (Type& c)
    {
      os << "class " << e << name;
    }

    virtual void
    inherits_none (Type&)
    {
      os << " : public ::XSCRT::Type"
         << "{"
         << "//@@ VC6 anathema" << endl
         << "typedef ::XSCRT::Type Base__;"
         << endl;
    }

    virtual void
    post (Type& c)
    {
      os << "public:" << endl;

      // c-tor (all-non-optional-members)
      //
      os << name << " (";

      ctor_args_.traverse (c);

      os <<  ");"
         << endl;

      // c-tor (Element const&)
      //
      os << name << " (" << xml_element_type << " const&);";

      // c-tor (Attribute const&)
      //
      bool r (true);

      struct Traverser : Traversal::Inherits,
                         Traversal::Complex
      {
        Traverser (bool& v)
            : v_ (v)
        {
          edge_traverser (*this);
          node_traverser (*this);
        }

        virtual void
        traverse (SemanticGraph::Complex& c)
        {
          if (has<Traversal::Element> (c) ||
              has<Traversal::Attribute> (c)) v_ = false;
          else
          {
            inherits (c, *this);
          }
        }

      private:
        bool& v_;
      } t (r);

      t.traverse (c);

      if (r)
      {
        os << name << " (" << xml_attribute_type << " const&);";
      }

      // copy c-tor
      //
      os << name << " (" << name << " const& s);"
         << endl;


      // operator=
      //
      os << name << "&" << endl
         << "operator= (" << name << " const& s);"
         << endl;

      os << "private:" << endl
         << "char regulator__;"
         << "};";
    }

  private:
    string name;
    string e;

    Traversal::Inherits inherits_;
    Traversal::Names names_;

    Base base_;
    Element element_;
    Attribute attribute_;

    CtorArgs ctor_args_;
  };


  struct Enumerator : Traversal::Enumerator, protected virtual Context
  {
    Enumerator (Context& c)
        : Context (c)
    {
    }

    virtual void
    traverse (Type& e)
    {
      os << "static " << scope << " const " << id (e.name ()) << ";";
    }
  };

  struct Label : Traversal::Enumerator, protected virtual Context
  {
    Label (Context& c)
        : Context (c)
    {
    }

    virtual void
    traverse (Type& e)
    {
      os << id (e.name ()) << "_l," << endl;
    }
  };

  struct Enumeration : Traversal::Enumeration, protected virtual Context
  {
    Enumeration (Context& c, string const& name_ = L"")
        : Context (c),
          name (name_),
          ex (c.esymbol.empty () ? c.esymbol : c.esymbol + L" "),
          enumerator_ (c),
          label_ (c)
    {
      names_enumerators_.node_traverser (enumerator_);
      names_labels_.node_traverser (label_);
    }

    virtual void
    traverse (Type& e)
    {
      if (e.named ()) name = id (e.name ());

      os << "class " << ex << name << " : public ::XSCRT::Type"
         << "{"
         << "public:" << endl;

      enter_scope (name);

      // c-tor (Element const&)
      //
      os << name << " (" << xml_element_type << " const&);";

      // c-tor (Attribute const&)
      //
      os << name << " (" << xml_attribute_type << " const&);";

      os << endl;

      names (e, names_enumerators_);

      os << endl;

      os << "enum Value"
         << "{";

      names (e, names_labels_);

      os << "};"
         << endl;

      os << "Value" << endl
         << "integral () const;"
         << endl;

      os << "friend bool" << endl
         << "operator== (" << name << " const& a, " << name << " const& b);"
         << endl;

      os << "friend bool" << endl
         << "operator!= (" << name << " const& a, " << name << " const& b);"
         << endl;


      os << "private:" << endl;

      os << name << " (Value v);"
         << endl;

      os << "Value v_;";

      leave_scope ();

      os << "};";
    }

  private:
    string name;
    string ex;

    Enumerator enumerator_;
    Traversal::Names names_enumerators_;

    Label label_;
    Traversal::Names names_labels_;
  };


  struct AnonymousType : Traversal::Element, protected virtual Context
  {
    AnonymousType (Context& c)
        : Context (c)
    {
    }

    virtual void
    traverse (Type& e)
    {
      SemanticGraph::Type& t (e.type ());

      if (!t.named () && !t.context ().count ("seen"))
      {
        string name (e.name ());

        os << "// anonymous type for " << scope << "::" << name << endl
           << "//" << endl;

        if (dynamic_cast<SemanticGraph::Type*> (&e.scope ()))
        {
          os << "public:" << endl;
        }

        Traversal::Belongs belongs;
        Complex complex (*this, *this, id (name));
        Enumeration enumeration (*this, id (name));

        belongs.node_traverser (complex);
        belongs.node_traverser (enumeration);

        t.context ().set ("seen", true);

        Element::belongs (e, belongs);

        t.context ().remove ("seen");
      }
    }
  };


  struct Includes : Traversal::Imports,
                    Traversal::Includes,
                    protected virtual Context
  {
    Includes (Context& c, std::string const& expr)
        : Context (c), expr_ (expr)
    {
    }

    virtual void
    traverse (SemanticGraph::Imports& i)
    {
      traverse (i.file ());
    }

    virtual void
    traverse (SemanticGraph::Includes& i)
    {
      traverse (i.file ());
    }

    virtual void
    traverse (fs::path const& f)
    {
      std::string name (regex::perl_s (f.string (), expr_));

      os << "#include \"" << name.c_str () << "\"" << endl
         << endl;
    }

  private:
    std::string expr_;
  };
}

void
generate_header (Context& ctx,
                 SemanticGraph::Schema& schema,
                 std::string const& expr)
{
  ctx.os << "#include <memory>" << endl
         << "#include <vector>" << endl
         << "#include <XMLSchema/Types.hpp>" << endl
         << endl;

  Traversal::Schema traverser;

  Traversal::Sources sources;
  Includes includes (ctx, expr);
  Traversal::Names schema_names;

  Namespace ns (ctx);

  traverser.edge_traverser (sources);
  traverser.edge_traverser (includes);
  traverser.edge_traverser (schema_names);
  sources.node_traverser (traverser);
  schema_names.node_traverser (ns);

  Traversal::Names names;

  AnonymousType anonymous_type (ctx);
  Complex complex (ctx, anonymous_type);
  Enumeration enumeration (ctx);

  ns.edge_traverser (names);
  names.node_traverser (complex);
  names.node_traverser (enumeration);
  names.node_traverser (anonymous_type);

  traverser.traverse (schema);
}
